/*
 * 	CLARK, CLAssifier based on Reduced K-mers.
 */

/*
 * 	This program is free software: you can redistribute it and/or modify
 *   	it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, either version 3 of the License, or
 *      (at your option) any later version.
 *   
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *           
 *      You should have received a copy of the GNU General Public License
 *      along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *              
 *      Copyright 2013-2019, Rachid Ounit <clark.ucr.help at gmail.com>
 *                
 */                 

/*
 *      @author: Rachid Ounit, Ph.D.
 *      @project: CLARK, Metagenomic and Genomic Sequences Classification project.
 *      @note: C++ IMPLEMENTATION supported on latest Linux and Mac OS.
 *     
 */


#include <cstdlib>
#include <vector>
#include <string>
#include <fstream>
#include <iostream>

#include "file.hh"
using namespace std;

int getSeqFileType(const char* _filename)
{
	FILE * fd = fopen(_filename,"r");
	if (fd == NULL)
	{	return -1;}
	string line = "";
	getLineFromFile(fd, line);
	
	if (line[0] == '@')
	{	
		char l1 = '1', l2 = '2', l4 = '4';
		getLineFromFile(fd, line);
		l1 = line[0];
		getLineFromFile(fd, line);
                l2 = line[0];
		getLineFromFile(fd, line);
		getLineFromFile(fd, line);
                l4 = line[0];
		if (l1 != '@' && l2 == '+' && l4 == '@')
		{	return 10;	}
		return 11;
	}
	fclose(fd);
	if (line[0] == '>')
        {       return 23; }
	return 100;
}

int getCLARKResFileType(const char* _filename)
{
        FILE * fd = fopen(_filename,"r");
        if (fd == NULL)
        {       return -1;}
        string line = "";
	vector<string> ele;
	vector<char> sep;
	sep.push_back(',');

        getLineFromFile(fd, line);
        fclose(fd);
	getElementsFromLine(line, sep, ele);
        if (ele.size() == 3)
        {       return 10; }
        if (ele.size() == 8)
        {       return 23; }
        return 100;
}

int main(int argc, const char** argv)
{
	if (argc < 5)
	{
		cerr << argv[0] << " <int: taxonomyID> <filename: input file of sequences> <filename: input file of the CLARK/CLARK-S results> <filename: output file to store the extracted sequences> <double: minGamma> <double: minConfidence>"<< endl;	
		cerr << " Notes: " << endl;
		cerr << " (1) The CLARK/CLARK-S results file should be the file produced from the input file of the sequences. " << endl;
		cerr << " (2) There is no need to put an extension for the output file generated by this program. It will be automatically set, it is the same format than the input file of the sequences. " << endl;
		cerr << " (3) The taxonomyID should make sense for the results file with respect to the taxonomy rank used. If the results file was created for example at the species level classification then the taxonomyID should be a taxon for that same rank (i.e., species)." << endl;
		exit(1);
	}	
	////////////////////////////////////////////////////////////////////////////////////////////////////	
	int seqType = getSeqFileType(argv[2]);
	if (seqType == -1)
	{
		cerr << "Failed to open: " << argv[2] << endl; exit(1);
	}
	bool isFastq = seqType == 10;
	bool isFasta = seqType == 23;
	
	if (!isFastq && !isFasta)
	{	
		cerr << "Failed to recognize the file format: It does not look like a fastq/fasta format."<< endl; exit(2); 
	}
	
	int CLARKreportType = getCLARKResFileType(argv[3]);	
	if (CLARKreportType == -1)
        {
                cerr << "Failed to open: " << argv[3] << endl; exit(1);
        }
	
	bool isShortReport = CLARKreportType == 10;
	//if (isShortReport) 
	//{ cerr << "Short report detected!" << endl; }
	bool isLongReport = CLARKreportType == 23;
	//if (isLongReport) 
	//{ cerr << "Long report detected!" << endl; }	

	if (!isLongReport && !isShortReport)
	{
		cerr << "Failed to recognize the results file format: It does not look like a CLARK results file." << endl; exit(2); 	
	}
	if (isLongReport && argc < 7)
	{
		cerr << "Please provide both the min gamma value and min confidence value for this result file." << endl; exit(3);
	}	
	///////////////////////////////////////////////////////////////////////////////////////////////////
	
	FILE * fd  = fopen(argv[2],"r");
	FILE * fdr = fopen(argv[3],"r"); 
	
	///////////////////
	string outputfile(argv[4]);
        if (isFastq)
        {       outputfile.push_back('.'); outputfile.push_back('f'); outputfile.push_back('q');}
        if (isFasta)
        {       outputfile.push_back('.'); outputfile.push_back('f'); outputfile.push_back('a');}

	FILE * fdt = fopen(outputfile.c_str(),"r");
	if (fdt == NULL)
	{
		fdt = fopen(outputfile.c_str(),"w");
		if (fdt == NULL)
		{
			cerr << "Failed to create/open output file: " << argv[4] << endl;
			exit(5);
		}
	}
	fclose(fdt);
	///////////////////
	string TaxidToChk(argv[1]);
	double minG = atof(argv[5]);
	double minC = atof(argv[6]);
	if (minG > 3.0)
	{
		cerr << "No sequence can be assigned with gamma score higher than 3.0 (which is the highest value achieved when using CLARK-S)."<< endl;
		exit(1);
	}
	if (minC > 1.0)
        {
        	cerr << "No sequence can be assigned with confidence score higher than 1.0."<< endl;
        	exit(1);
	}
	string line = "";
	vector<string> ele;
	vector<char> sep;
	sep.push_back(',');
	vector<string> IDs;
	// Passing header
	getLineFromFile(fdr, line);
	//Object_ID,Length,Gamma,1st_assignment,score1,2nd_assignment,score2,confidence
	int idxToChk = isLongReport ? 3: 2; 
	if (isLongReport)
	{
		cerr << "Info: The program will extract all sequences mapped to taxid:" << argv[1] ;
		cerr << " with confidence score higher than " << minC << ", and gamma score higher than " << minG << "." << endl;
		while (getLineFromFile(fdr, line))
		{
			ele.clear();
			getElementsFromLine(line, sep, ele);
			if (ele[idxToChk] == TaxidToChk && atof(ele[2].c_str())> minG && atof(ele[7].c_str())> minC)
			{
				IDs.push_back(ele[0]);
			}
		}
	}
	else
	{
		cerr << "Info: The program will extract all sequences mapped to taxid:" << argv[1] << "." << endl;
		while (getLineFromFile(fdr, line))
                {
                        ele.clear();
                        getElementsFromLine(line, sep, ele);
                        if (ele[idxToChk] == TaxidToChk) 
	                {
                                IDs.push_back(ele[0]);
                        }
                }
	}
	fclose(fdr);
	if (IDs.size() < 1)
	{
		cerr << "There is no sequence assigned to taxid:" << TaxidToChk ;	
		if (isLongReport)
		{
			cerr <<" passing the filtering requested" ;
		}
		cerr << "." << endl;
		return 0;
	}
	else
	{
		cerr << IDs.size() << " sequences mapping to " << TaxidToChk << " have been identified..." << endl;
	}

	ofstream fout(outputfile.c_str());

	sep.clear();
	sep.push_back(' ');
	sep.push_back('\t');
	int cpt  = 0;
	string line1= "", line2= "", line3="", line4="";
	if (isFastq)
	{
		while (cpt < IDs.size())
		{
			getLineFromFile(fd, line1);
			if (line1[0] == '@')
			{
				line = line1.substr(1,line1.size());
				ele.clear();
				getElementsFromLine(line, sep, ele);
				getLineFromFile(fd, line2);
				getLineFromFile(fd, line3);
				getLineFromFile(fd, line4);
				if (ele[0] == IDs[cpt])
				{
					cpt++;
					fout << line1 << "\n" << line2 << "\n" << line3 << "\n" << line4 << endl;
				}
			}
		}
	}
	else
	{
		bool ToContinue =  false;
		while (cpt < IDs.size() || ToContinue)
                {
			getLineFromFile(fd, line1);
                        if (line1[0] == '>')
                        {
				ToContinue = false;
				line = line1.substr(1,line1.size());
                                ele.clear();
                                getElementsFromLine(line, sep, ele);
                                if (cpt < IDs.size() && ele[0] ==IDs[cpt])
                                {
					ToContinue = true;
					cpt++;
                                        fout << line1 << endl;
				}
                        }
			else if (ToContinue)
			{
				fout << line1 << endl;
			}
                }
	}
	fclose(fd);
	fout.close();
	return 0;
}


